﻿using Hims.Shared.UserModels.Common;

namespace Hims.Api.Controllers
{
    using System;
    using System.Collections.Generic;
    using System.Threading.Tasks;
    using Domain.Services;
    using Hims.Api.Models;
    using Hims.Shared.DataFilters;
    using Hims.Shared.Library.Enums;
    using Hims.Shared.UserModels;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Npgsql;
    using Shared.EntityModels;
    using Utilities;

    /// <inheritdoc />
    [Authorize]
    [Route("api/cubicle")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class CubicleController : BaseController
    {

        /// <summary>
        /// The auditlog services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        /// <summary>
        /// The admission services.
        /// </summary>
        private readonly ICubicleService cubicleService;

        /// <inheritdoc />
        public CubicleController(
            IAuditLogService auditLogServices,
            ICubicleService cubicleService)
        {

            this.auditLogServices = auditLogServices;
            this.cubicleService = cubicleService;
        }

        /// <summary>
        /// The fetch cubicles.
        /// </summary>
        /// <param name="model">
        /// The widget filter model.
        /// </param>
        /// <returns>
        /// The list of cubicle.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - List of cubicles.
        /// - 500 - Problem with Server side code.
        /// </remarks>

        [HttpPost]
        [Route("fetch")]
        public async Task<ActionResult> FetchAsync([FromBody] CubicleFilterModel model)
        {
            try
            {
                var cubicles = await this.cubicleService.FetchAsync(model).ConfigureAwait(false);
                return Ok(new GenericResponse
                {
                    Status = GenericStatus.Success,
                    Data = cubicles
                });
            }
            catch (Exception ex)
            {
                return Ok(new GenericResponse
                {
                    Status = GenericStatus.Error,
                    Message = ex.Message
                });
            }
        }

        /// <summary>
        /// The insert cubicle async.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("insert-cubicle")]
        public async Task<ActionResult> InsertAsync([FromBody] CubicleModel model, [FromHeader] LocationHeader header)
        {
            try
            {
                var response = await this.cubicleService.InsertAsync(model).ConfigureAwait(false);
                if (response.Status == GenericStatus.Info)
                {
                    return Ok(new GenericResponse
                    {
                        Status = GenericStatus.Info,
                        Message = "Given Cubicle already exists."
                    });
                }
                if (response.Status == GenericStatus.Success)
                {
                    var auditLogModel = new AuditLogModel
                    {
                        AccountId = model.CreatedBy,
                        LogTypeId = (int)LogTypes.Cubicle,
                        LogFrom = (short)model.LogFrom,
                        LogDate = DateTime.UtcNow.AddMinutes(330),
                        LogDescription = $@"<b>{model.CreatedByName}</b> has <b>added</b> a new cubicle <b>{model.Name}</b> on {DateTime.UtcNow.AddMinutes(330)}",
                        LocationId = Convert.ToInt32(header.LocationId)
                    };
                    await this.auditLogServices.LogAsync(auditLogModel);
                }

                return Ok(response);
            }
            catch (Exception ex)
            {
                return Ok(new GenericResponse
                {
                    Status = GenericStatus.Error,
                    Message = ex.Message
                });
            }
        }

        /// <summary>
        /// The update widget async.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("update-cubicle")]
        public async Task<ActionResult> UpdateAsync([FromBody] CubicleModel model, [FromHeader] LocationHeader header)
        {
            try
            {
                var response = await this.cubicleService.UpdateAsync(model).ConfigureAwait(false);

                if (response.Status == GenericStatus.Info)
                {
                    return Ok(new GenericResponse
                    {
                        Status = GenericStatus.Info,
                        Message = "Assigned Cubicle cannot be edited."
                    });
                }

                if (response.Status == GenericStatus.Success)
                {
                    var auditLogModel = new AuditLogModel
                    {
                        AccountId = model.ModifiedBy,
                        LogTypeId = (int)LogTypes.Cubicle,
                        LogFrom = (short)model.LogFrom,
                        LogDate = DateTime.UtcNow.AddMinutes(330),
                        LogDescription = $@"<b>{model.ModifiedByName}</b> has <b>updated</b> cubicle name <b>{model.Name}</b> on {DateTime.UtcNow.AddMinutes(330)}",
                        LocationId = Convert.ToInt32(header.LocationId)
                    };
                    await this.auditLogServices.LogAsync(auditLogModel);
                }
                return Ok(response);
            }
            catch (Exception ex)
            {
                return Ok(new GenericResponse
                {
                    Status = GenericStatus.Error,
                    Message = ex.Message
                });
            }
        }

        /// <summary>
        /// The update widget async.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("assign")]
        public async Task<ActionResult> AssignAsync([FromBody] CubicleModel model, [FromHeader] LocationHeader header)
        {
            try
            {
                var response = await this.cubicleService.AssignAsync(model).ConfigureAwait(false);
                return Ok(new GenericResponse
                {
                    Status = response == -1 ? GenericStatus.Warning : response == -2 ? GenericStatus.Info : GenericStatus.Success
                });
            }
            catch (Exception ex)
            {
                return Ok(new GenericResponse
                {
                    Status = GenericStatus.Error,
                    Message = ex.Message
                });
            }
        }

        /// <summary>
        /// The update widget async.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("assign-consultant-doctor")]
        public async Task<ActionResult> AssignConsultDoctorAsync([FromBody] CubicleModel model, [FromHeader] LocationHeader header)
        {
            try
            {
                var response = await this.cubicleService.AssignConsultantDoctorAsync(model).ConfigureAwait(false);
                return Ok(new GenericResponse
                {
                    Status = GenericStatus.Success,
                    Data = response
                });
            }
            catch (Exception ex)
            {
                return Ok(new GenericResponse
                {
                    Status = GenericStatus.Error,
                    Message = ex.Message
                });
            }
        }
        /// <summary>
        /// The update widget async.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("unassign-consultant-doctor")]
        public async Task<ActionResult> UnassignConsultDoctorAsync([FromBody] CubicleModel model, [FromHeader] LocationHeader header)
        {
            try
            {
                var response = await this.cubicleService.UnAssignConsultantDoctorAsync(model).ConfigureAwait(false);
                return Ok(new GenericResponse
                {
                    Status = GenericStatus.Success
                });
            }
            catch (Exception ex)
            {
                return Ok(new GenericResponse
                {
                    Status = GenericStatus.Error,
                    Message = ex.Message
                });
            }
        }

        /// <summary>
        /// The update widget async.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("unassign")]
        public async Task<ActionResult> UnassignAsync([FromBody] CubicleModel model, [FromHeader] LocationHeader header)
        {
            try
            {
                var response = await this.cubicleService.UnassignAsync(model).ConfigureAwait(false);
                return Ok(new GenericResponse
                {
                    Status = response == -1 ? GenericStatus.Warning : GenericStatus.Success
                });
            }
            catch (Exception ex)
            {
                return Ok(new GenericResponse
                {
                    Status = GenericStatus.Error,
                    Message = ex.Message
                });
            }
        }

        /// <summary>
        /// The deactivate widget async.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("update-cubicle-status")]
        public async Task<ActionResult> DeactivateAsync([FromBody] CubicleModel model, [FromHeader] LocationHeader header)
        {
            try
            {
                var response = await this.cubicleService.UpadteStatusAsync((int)model.CubicleId, model.ActiveStatusId, (int)model.ModifiedBy);

                if (response > 0)
                {
                    var auditLogModel = new AuditLogModel
                    {
                        AccountId = model.ModifiedBy,
                        LogTypeId = (int)LogTypes.Cubicle,
                        LogFrom = (short)model.LogFrom,
                        LogDate = DateTime.UtcNow.AddMinutes(330),
                        LogDescription = $@"<b>{model.ModifiedByName}</b> has <b>updated</b> cubicle status on {DateTime.UtcNow.AddMinutes(330)}",
                        LocationId = Convert.ToInt32(header.LocationId)
                    };
                    await this.auditLogServices.LogAsync(auditLogModel);
                }

                return Ok(new GenericResponse
                {
                    Status = response switch
                    {
                        -1 => GenericStatus.Error,
                        -2 => GenericStatus.Info,
                        _ => GenericStatus.Success
                    },
                    Data = response
                });
            }
            catch (Exception ex)
            {
                return Ok(new GenericResponse
                {
                    Status = GenericStatus.Error,
                    Message = ex.Message
                });
            }
        }

        /// <summary>
        /// The delete specialization.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Specialization deleted successfully.
        /// - 409 - Specialization can not be deleted.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("delete")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> DeleteAsync([FromBody] CubicleModel model, [FromHeader] LocationHeader header)
        {
            try
            {
                model = (CubicleModel)EmptyFilter.Handler(model);
                var response = await this.cubicleService.DeleteAsync(model.CubicleId);
                if (response == 0)
                {
                    response = -2;
                }
                if (response > 0)
                {
                    var auditLogModel = new AuditLogModel
                    {
                        AccountId = model.ModifiedBy,
                        LogTypeId = (int)LogTypes.Cubicle,
                        LogFrom = (int)AccountType.Administrator,
                        LogDate = DateTime.UtcNow.AddMinutes(330),
                        LogDescription = $" <b>{model.ModifiedByName}</b> has <b>deleted</b> the <b>{model.Name}</b> on {DateTime.UtcNow.AddMinutes(330)}",
                        LocationId = Convert.ToInt32(header.LocationId)
                    };
                    await this.auditLogServices.LogAsync(auditLogModel);
                }

                return Ok(new GenericResponse
                {
                    Status = response switch
                    {
                        -1 => GenericStatus.Error,
                        -2 => GenericStatus.Info,
                        _ => GenericStatus.Success
                    },
                    Data = response
                });
            }
            catch (Exception ex)
            {
                return Ok(new GenericResponse
                {
                    Status = GenericStatus.Error,
                    Message = ex.Message
                });
            }
        }
    }
}